<?php

namespace App\Admin\Controllers;

use OpenAdmin\Admin\Controllers\AdminController;
use OpenAdmin\Admin\Form;
use OpenAdmin\Admin\Grid;
use OpenAdmin\Admin\Show;
use \App\Models\PageSection;
use \App\Models\SectionTemplate;

class PageSectionController extends AdminController
{
    /**
     * Title for current resource.
     *
     * @var string
     */
    protected $title = 'PageSection';

    /**
     * Make a grid builder.
     *
     * @return Grid
     */
    protected function grid()
    {
        $grid = new Grid(new PageSection());

        $grid->filter(function($filter){
            // Remove the default id filter
            $filter->disableIdFilter();
            $filter->like('section_name', 'Section Name');
            $filter->equal('page_id', 'Page')->select(\App\Models\Page::pluck('name', 'id'));
        });

        $grid->column('id', __('Id'));
        $grid->column('hint_image', __('Hint Image'))->display(function () {
            return "<img src='" . asset('section_hint_image/' . $this->section_name.'.png') . "' style='max-width:200px;max-height:100px' />";
        });
        $grid->column('page.name', __('Page'));
        $grid->column('section_name', __('Section name'))->label();
        $grid->column('created_at', __('Created at'))->display(function ($value) {
            return \Carbon\Carbon::parse($value)->format('d M Y, H:i');
        });
        $grid->column('updated_at', __('Updated at'))->display(function ($value) {
            return \Carbon\Carbon::parse($value)->format('d M Y, H:i');
        });

        $grid->actions(function ($actions) {
            // $actions->disableDelete(); // Removes the delete button
            // $actions->disableEdit();   // Removes the edit button
            $actions->disableView();   // Removes the view button
        });

        return $grid;
    }

    /**
     * Make a show builder.
     *
     * @param mixed $id
     * @return Show
     */
    protected function detail($id)
    {
        $show = new Show(PageSection::findOrFail($id));

        $show->field('id', __('Id'));
        $show->field('page_id', __('Page id'));
        $show->field('hint_image', __('Hint image'));
        $show->field('section_name', __('Section name'))->label();
        $show->field('section_json', __('Section json'));
        $show->field('created_at', __('Created at'));
        $show->field('updated_at', __('Updated at'));

        return $show;
    }

    /**
     * Make a form builder.
     *
     * @return Form
     */

    protected function form()
    {
        $form = new Form(new PageSection());

        // Disable the delete button on the edit form
        $form->tools(function (Form\Tools $tools) {
            $tools->disableDelete(); // Hides the delete button from the tools section
            $tools->disableView();   // Hides the view button from the tools section
            // $tools->disableList();  // Hides the list button
        });

        if($form->isCreating()) {
            $form->select('page_id', 'Page')
                ->options(
                    \App\Models\Page::pluck('name', 'id')
                )
                ->default(request('page_id') ?? null)
                ->required();
            $form->select('section_name', 'Type')
                ->options([
                    'home_services_section' => 'Home Services Section',
                    'home_hero_section' => 'Home Hero Section',
                    'home_choosing_topfilings_section' => 'Home Choosing Topfilings Section',
                    'home_counter_section' => 'Home Counter Section',
                    'home_verified_section' => 'Home Verified Section',
                    'home_blog_section' => 'Blog Section',
                    'home_testimonial_section' => 'Home Testimonial Section',
                    'service_content_form_section' => 'Service Content + Form Section',
                    'service_why_section' => 'Service Why Section',
                    'service_icon_box' => 'Service Icon Box Section',
                    'service_business_structure' => 'Service Business Structure Chart Section',
                    'parallax_section' => 'Parallax Section',
                    'two_column_list' => 'Two Column List',
                    'image_content_section' => 'Image Content Section',
                    'virtual_cfo_why_section' => 'Virtual CFO Why Section',
                    'content' => 'Content Section',
                    'title_paragraph_card' => 'Title Paragraph Card',
                ])
                ->help('First create the section with type, then edit to add content.')
                ->required();
        }

        $form->editing(function (Form $form) {
            $form->divider($form->model()->section_name);
            if ($form->model()->hint_image) {
                $form->html('<img src="' . asset('storage/'.$form->model()->hint_image) . '" style="max-width:400px;" />', 'Current Hint Image');
            }
            // hero section -----------------------------------------------------------------------------
            if ($form->model()->section_name == 'home_hero_section') {
                $sectionJson = $form->model()->section_json;
                $form->embeds('section_json', 'Hero Section', function ($json) use($sectionJson) {
                    $json->textarea('sub_heading', 'Sub-heading')->required();
                    $json->text('heading', 'Heading')->required();
                    $json->textarea('paragraph', 'Paragraph')->required();
                    $json->multipleImage('images', 'Images')
                        ->uniqueName()
                        ->move('sections')
                        ->help('Upload images for the hero section')
                        ->value($sectionJson['images'] ?? null);
                    $json->hidden('current_images', 'Current Images')
                        ->default(isset($sectionJson['images']) ? json_encode($sectionJson['images']) : null)
                        ->readonly();
                });
            }
            // service section ---------------------------------------------------------------------------
            if ($form->model()->section_name == 'home_services_section') {
                $sectionJson = $form->model()->section_json;
                $form->embeds('section_json', 'Service Section', function ($json) use($sectionJson) {
                    $json->text('heading', 'Section Heading')->required();
                    $json->textarea('sub_heading', 'Sub-heading')->required();
                    
                    $json->embeds('cards', 'Cards', function ($json) use($sectionJson) {
                        for ($i = 0; $i < 3; $i++) {
                            $json->embeds($i, 'Card-'.$i+1, function ($json) use ($i, $sectionJson) {
                                $json->text('title', 'Title')->required();
                                $json->textarea('description', 'Description')->required();
                                $json->image('image', 'Image')
                                    ->uniqueName()
                                    ->move('sections')
                                    ->help('Upload card image')
                                    ->value($sectionJson['cards'][$i]['image'] ?? null);
                                if(isset($sectionJson['cards'][$i]['image'])) {
                                    $json->hidden('current_image', 'Current Image')
                                        ->default($sectionJson['cards'][$i]['image'])
                                        ->readonly();
                                }
                                
                                $json->url('link', 'Link')->placeholder('https://example.com');
                            });
                        }
                    });
                });
            }
            //choosing_topfilings section ---------------------------------------------------------------------------
            if ($form->model()->section_name == 'home_choosing_topfilings_section') {
                $sectionJson = $form->model()->section_json;
                $form->embeds('section_json', 'Choosing TopFilings Section', function ($json) use($sectionJson) {
                    $json->text('heading', 'Section Heading')->required();
                    $json->textarea('sub_heading', 'Sub-heading')->required();
                    $json->embeds('menus', 'Menus', function ($json) use($sectionJson) {
                        for ($i = 0; $i < 4; $i++) {
                            $json->embeds($i, 'Menu-'.$i+1, function ($json) use ($i, $sectionJson) {
                                $json->text('title', 'Title')->required();
                                $json->textarea('paragraph', 'Paragraph')->required();
                                $json->image('image', 'Image')
                                    ->uniqueName()
                                    ->move('sections')
                                    ->help('Upload menu image')
                                    ->value($sectionJson['cards'][$i]['image'] ?? null);
                                if(isset($sectionJson['menus'][$i]['image'])) {
                                    $json->hidden('current_image', 'Current Image')
                                        ->default($sectionJson['menus'][$i]['image'])
                                        ->readonly();
                                }
                            });
                        }
                    });
                });
            }
            // counter_section ---------------------------------------------------------------------------
            if ($form->model()->section_name == 'home_counter_section') {
                $form->embeds('section_json', 'Counter Section', function ($json){
                    $json->embeds('counters', 'Counters', function ($json) {
                        for ($i = 0; $i < 3; $i++) {
                            $json->embeds($i, 'Counter-'.$i+1, function ($json) {
                                $json->text('title', 'Title')->required();
                                $json->text('paragraph', 'Paragraph')->required();
                                $json->text('count', 'Count')->required();
                                $json->text('suffix', 'Suffix')->required();
                            });
                        }
                    });
                });
            }
            // verified_section ---------------------------------------------------------------------------
            if ($form->model()->section_name == 'home_verified_section') {
                $sectionJson = $form->model()->section_json;
                $form->embeds('section_json', 'Verified Section', function ($json) use($sectionJson) {
                    $json->text('heading', 'Section Heading')->required();
                    $json->image('image', 'Image')
                        ->uniqueName()
                        ->move('sections')
                        ->help('Upload image for the section')
                        ->value($sectionJson['image'] ?? null);
                    if(isset($sectionJson['image'])) {
                        $json->hidden('current_image', 'Current Image')
                            ->default($sectionJson['image'])
                            ->readonly();
                    }
                    $json->embeds('lists', 'Lists', function ($json) use($sectionJson) {
                        for ($i = 0; $i < 4; $i++) {
                            $json->embeds($i, 'List-'.$i+1, function ($json) use ($i, $sectionJson) {
                                $json->text('title', 'Title')->required();
                                $json->textarea('paragraph', 'Paragraph')->required();
                                $json->image('icon', 'Icon')
                                    ->uniqueName()
                                    ->move('sections')
                                    ->help('Upload icon image')
                                    ->value($sectionJson['lists'][$i]['icon'] ?? null);
                                if(isset($sectionJson['lists'][$i]['icon'])) {
                                    $json->hidden('current_icon', 'Current Icon')
                                        ->default($sectionJson['lists'][$i]['icon'])
                                        ->readonly();
                                }
                                $json->textarea('form_paragraph', 'Form Paragraph')->required();
                                $json->text('button_text', 'Form Button Text')->required();
                            });
                        }
                    });
                });
            }
            // home page blog section ---------------------------------------------------------------------------
            if($form->model()->section_name == 'home_blog_section') {
                $form->embeds('section_json', 'Blog Section', function ($json) {
                    $json->text('heading', 'Section Heading')->required();
                    $json->textarea('sub_heading', 'Sub-heading')->required();
                });
            }
            // home page Testimonial section ---------------------------------------------------------------------------
            if($form->model()->section_name == 'home_testimonial_section') {
                $form->embeds('section_json', 'Testimonial Section', function ($json) {
                    $json->text('heading', 'Section Heading')->required();
                    $json->textarea('sub_heading', 'Sub-heading')->required();
                });
            }
            // service_content_form_section ---------------------------------------------------------------------------
            if($form->model()->section_name == 'service_content_form_section') {
                $sectionJson = $form->model()->section_json;
                $form->embeds('section_json', 'Service Content + Form Section', function ($json) use($sectionJson) {
                    $json->text('sub_heading', 'Sub-heading');
                    $json->text('heading', 'Section Heading')->required();
                    $json->textarea('paragraph', 'Paragraph');
                    for($i=0; $i<5; $i++){
                        $json->text("list_$i", "List ".$i+1);
                    }
                    //form fields
                    $json->divider('Form Details');
                    $json->text('form_heading', 'Form Heading')->required();
                    $json->textarea('form_sub_heading', 'Form Sub-heading')->required();
                    $json->text('form_button_text', 'Form Button Text')->required();
                });
            }
            // service_why_section ----------------------------------------------------------------
            if ($form->model()->section_name == 'service_why_section') {
                $sectionTable = $form->model()->section_table;
                $form->html(
                    '<style>
                        #has-many-section_table{
                            overflow-x: auto;
                            width: 900px;
                        }
                        #has-many-section_table table td{
                            min-width: 400px;
                        }
                        #has-many-section_table table td.hidden{
                            min-width: 10px;
                       }
                    }
                    </style>'
                );
                $form->embeds('section_json', 'Service Why Section', function ($json) {
                    $json->text('sub_heading', 'Sub-heading');
                    $json->text('heading', 'Section Heading')->required();
                });
                $form->table('section_table', 'Cards', function ($table) {
                    $table->text('title', 'Title')
                        ->required();
                    $table->image('icon', 'Icon')
                        ->uniqueName()
                        ->move('sections')
                        ->help('Upload card icon (Recommended size: 64x64px)')
                        ->value($table->icon ?? null);
                    $table->hidden('current_icon', 'Current Icon')->value($table->icon ?? null);
                    $table->textarea('paragraph', 'Paragraph')
                        ->required();
                })->default($sectionTable);
            }
            // service_icon_box ----------------------------------------------------------------
            if ($form->model()->section_name == 'service_icon_box') {
                $sectionTable = $form->model()->section_table;
                $form->html(
                    '<style>
                        #has-many-section_table{
                            overflow-x: auto;
                            width: 900px;
                            min-height: 400px;
                        }
                        #has-many-section_table table td{
                            min-width: 400px;
                        }
                        #has-many-section_table table td.hidden{
                            min-width: 10px;
                       }
                    }
                    </style>'
                );
                $form->embeds('section_json', 'Service Icon Box', function ($json) {
                    $json->text('sub_heading', 'Sub Heading');
                    $json->text('heading', 'Section Heading')->required();
                });
                $form->table('section_table', 'Cards', function ($table) use($form) {
                    $table->text('icon', 'Icon class (FontAwesome)')
                        ->help('for example: fa-solid fa-cog')
                        ->value($table->icon ?? null);
                    $table->text('title', 'Title')
                        ->required();
                    $table->hidden('current_icon', 'Current Icon')->value($table->icon ?? null);
                    $table->textarea('paragraph', 'Paragraph')
                        ->required();
                })->default($sectionTable);
            }
            // service_business_structure ----------------------------------------------------------------
            if ($form->model()->section_name == 'service_business_structure') {
                $form->html(
                    '<style>
                        #has-many-section_table{
                            overflow-x: auto;
                            width: 900px;
                            min-height: 400px;
                        }
                        #has-many-section_table table td{
                            min-width: 200px;
                        }
                        #has-many-section_table table td.hidden{
                            min-width: 10px;
                        }
                    }
                    </style>'
                );
                $form->embeds('section_json', 'Business Structure Section', function ($json) {
                    $json->text('sub_heading', 'Sub-heading')->required();
                    $json->text('heading', 'Section Heading')->required();
                });
                $form->table('section_table', 'Structures', function ($table) {
                    $table->text('service', 'Service')->required();
                    $table->text('private_limited_company', 'Private Limited Company')->required();
                    $table->text('one_person_company', 'One Person Company')->required();
                    $table->text('limited_liability_partnership', 'Limited Liability Partnership')->required();
                    $table->text('partnership_firm', 'Partnership Firm')->required();
                    $table->text('proprietorship_firm', 'Proprietorship Firm')->required();
                })->help('Add business structure details in tabular format');
            }
            // parallax_section ----------------------------------------------------------------
            if( $form->model()->section_name == 'parallax_section') {
                $sectionJson = $form->model()->section_json;
                $form->embeds('section_json', 'Parallax Section', function ($json) use($sectionJson) {
                    $json->text('heading', 'Section Heading')->required();
                    $json->textarea('paragraph', 'Paragraph')->required();
                    $json->image('background_image', 'Background Image')
                        ->uniqueName()
                        ->move('sections')
                        ->help('Upload background image for the parallax section')
                        ->value($sectionJson['background_image'] ?? null);
                    if(isset($sectionJson['background_image'])) {
                        $json->hidden('current_background_image', 'Current Background Image')
                            ->default($sectionJson['background_image'])
                            ->readonly();
                    }
                });
            }
            // two_column_list --------------------------------------------------------------------
            if( $form->model()->section_name == 'two_column_list') {
                $sectionJson = $form->model()->section_json;
                $form->embeds('section_json', 'Two Column List', function ($json) {
                    $json->text('heading', 'Section Heading')->required();
                    $json->text('sub_heading', 'Sub Heading');
                    $json->embeds('list_1', 'List 1', function ($json) {
                        $json->text('heading', 'Section Heading');
                        for($i=0; $i<5; $i++){
                            $json->text('item_'.$i, 'Item '.$i+1);
                        }
                    });
                    $json->embeds('list_2', 'List 2', function ($json) {
                        $json->text('heading', 'Section Heading');
                        for($i=0; $i<5; $i++){
                            $json->text('item_'.$i, 'Item '.$i+1);
                        }
                    });
                });
            }
            // image_content_section --------------------------------------------------------------
            if( $form->model()->section_name == 'image_content_section') {
                $form->html(
                    '<style>
                        #has-many-section_table{
                            overflow-x: auto;
                            width: 900px;
                        }
                        #has-many-section_table table td{
                            min-width: 300px;
                        }
                        #has-many-section_table table td textarea{
                            min-width: 500px;
                        }
                        #has-many-section_table table td.hidden{
                            min-width: 10px;
                       }
                    }
                    </style>'
                );
                $sectionTable = $form->model()->section_table;
                $form->table('section_table', 'Image Content Section', function ($table) {
                    $table->text('title', 'Title')
                        ->required();
                    $table->textarea('paragraph', 'Paragraph');
                    $table->image('image', 'Image')
                        ->uniqueName()
                        ->move('sections')
                        ->value($table->image ?? null);
                    $table->hidden('current_image', 'Current Image')->value($table->image ?? null);
                })->default($sectionTable);
            }
            // virtual_cfo_why_section ----------------------------------------------------------------
            if ($form->model()->section_name == 'virtual_cfo_why_section') {
                $sectionTable = $form->model()->section_table;
                $form->html(
                    '<style>
                        #has-many-section_table{
                            overflow-x: auto;
                            width: 900px;
                        }
                        #has-many-section_table table td{
                            min-width: 400px;
                        }
                        #has-many-section_table table td.hidden{
                            min-width: 10px;
                       }
                    }
                    </style>'
                );
                $form->embeds('section_json', 'Service Why Section', function ($json) {
                    $json->text('sub_heading', 'Sub-heading');
                    $json->text('heading', 'Section Heading')->required();
                });
                $form->table('section_table', 'Cards', function ($table) {
                    $table->text('title', 'Title')
                        ->required();
                    $table->image('icon', 'Icon')
                        ->uniqueName()
                        ->move('sections')
                        ->help('Upload card icon (Recommended size: 64x64px)')
                        ->value($table->icon ?? null);
                    $table->hidden('current_icon', 'Current Icon')->value($table->icon ?? null);
                    $table->textarea('paragraph', 'Paragraph')
                        ->required();
                })->default($sectionTable);
            }
            // content section ----------------------------------------------------------------
            if ($form->model()->section_name == 'content') {
                $form->embeds('section_json', 'Content', function ($json) {
                    $json->text('sub_heading', 'Sub-heading');
                    $json->text('heading', 'Section Heading')->required();
                    $json->ckeditor('content');
                });
            }
            // title_paragraph_card -----------------------------------------------------------
            if ($form->model()->section_name == 'title_paragraph_card') {
                $form->html('<style>
                    .cke_contents.cke_reset{height:300px!important;}
                </style>');
                $form->embeds('section_json', 'Title Paragraph Cards', function ($json) {
                    $json->text('heading', 'Section Heading')->required();
                    for($i=0; $i<8; $i++){
                        $json->ckeditor('card_'.$i, 'Card '.$i+1);
                    }
                });
            }

        });

        $form->saving(function (Form $form) {
            // On create, fetch hint_image and section_json from SectionTemplate
            if ($form->isCreating() && $form->section_name) {
                $template = SectionTemplate::where('section_name', $form->section_name)->first();
                if ($template) {
                    $form->model()->hint_image = $template->hint_image;
                    $form->model()->section_json = $template->section_json;
                }
            }
            $form->model()->section_json = request('section_json');

            // If section_table exists in request, set it to model
            if (null !== request('section_table')) {
                $form->model()->section_table = request('section_table');
            }
        });

        return $form;
    }

}
